/**
 * A class for generating and printing text patterns.
 * 
 * @author Hendrik Speleers
 * @author NMCGJ, AY 2025-2026
 */
public class TextPattern {

   protected int nbPrinters;
   protected int maxPrinters;
   protected TextPrinter[] printers;

   /**
    * Constructs a text pattern generator.
    */
   public TextPattern() {
      nbPrinters = 0;
      maxPrinters = 10;
      printers = new TextPrinter[maxPrinters];
   }

   /**
    * Adds a printer to the text pattern generator.
    * 
    * @param printer
    *           the text printer
    */
   public void addPrinter(TextPrinter printer) {
      if (printer != null && nbPrinters < maxPrinters) {
         printers[nbPrinters] = printer;
         nbPrinters++;
      }
   }

   /**
    * Sends a given text to all the printers.
    * 
    * @param text
    *           the given text
    */
   public void print(String text) {
      for (int i = 0; i < nbPrinters; i++) {
         printers[i].print(text);
      }
   }

   /**
    * Sends a given text to all the printers and then terminates the line.
    * 
    * @param text
    *           the given text
    */
   public void println(String text) {
      for (int i = 0; i < nbPrinters; i++) {
         printers[i].println(text);
      }
   }

   /**
    * Terminates the line on all the printers.
    */
   public void println() {
      for (int i = 0; i < nbPrinters; i++) {
         printers[i].println();
      }
   }

   /**
    * Sends a square text pattern to all the printers.
    * 
    * @param height
    *           the height of the square
    */
   public void printSquare(int height) {
      for (int i = 1; i <= height; i++) {
         for (int j = 1; j <= height; j++) {
            print("x");
         }
         println();
      }
   }

   /**
    * Sends a triangle text pattern to all the printers.
    * 
    * @param height
    *           the height of the triangle
    */
   public void printTriangle(int height) {
      for (int i = 1; i <= height; i++) {
         for (int j = 1; j <= i; j++) {
            print("x");
         }
         println();
      }
   }

   /**
    * Sends a pyramid text pattern to all the printers.
    * 
    * @param height
    *           the height of the pyramid
    */
   public void printPyramid(int height) {
      for (int i = 1; i <= height; i++) {
         for (int j = 1; j <= (height - i); j++) {
            print(" ");
         }
         for (int j = 1; j <= (2 * i - 1); j++) {
            print("x");
         }
         println();
      }
   }

   /**
    * A small test of the TextPattern class.
    */
   public static void main(String[] args) {
      // Construct the text frame
      TextFrame textFrame = new TextFrame("My Text Pattern Frame");
      textFrame.setTextEditable(false);
      textFrame.start();

      // Construct the printers
      ConsolePrinter consolePrinter = new ConsolePrinter();
      TextFramePrinter framePrinter = new TextFramePrinter(textFrame);

      // Construct a text pattern generator and add its printers
      TextPattern pattern = new TextPattern();
      pattern.addPrinter(consolePrinter);
      pattern.addPrinter(framePrinter);

      // Print some text patterns
      pattern.println("a triangle:");
      pattern.printTriangle(5);
      pattern.println();
      pattern.println("a square:");
      pattern.printSquare(5);
      pattern.println();
      pattern.println("a pyramid:");
      pattern.printPyramid(5);
   }

}
