import java.util.*;

/**
 * A class for generating and printing text patterns.
 * 
 * @author Hendrik Speleers
 * @author NMCGJ, AY 2025-2026
 */
public class TextPattern {

   protected List<TextPrinter> printers;

   /**
    * Constructs a text pattern generator.
    */
   public TextPattern() {
      printers = new ArrayList<TextPrinter>(10);
   }

   /**
    * Adds a printer to the text pattern generator.
    * 
    * @param printer
    *           the text printer
    */
   public void addPrinter(TextPrinter printer) {
      if (printer != null) {
         printers.add(printer);
      }
   }

   /**
    * Sends a given text to all the printers.
    * 
    * @param text
    *           the given text
    */
   public void print(String text) {
      for (TextPrinter printer : printers) {
         printer.print(text);
      }
   }
   
/*
   // ALTERNATIVE 1: using standard for loop
   public void print(String text) {
      for (int i = 0; i < printers.size(); i++) {
         printers.get(i).print(text);
      }
   }
*/

/*
   // ALTERNATIVE 2: using iterator
   public void print(String text) {
      Iterator<TextPrinter> iter = printers.iterator();
      while (iter.hasNext()) {
         iter.next().print(text);
      }
   }
*/

   /**
    * Sends a given text to all the printers and then terminates the line.
    * 
    * @param text
    *           the given text
    */
   public void println(String text) {
      for (TextPrinter printer : printers) {
         printer.println(text);
      }
   }

   /**
    * Terminates the line on all the printers.
    */
   public void println() {
      for (TextPrinter printer : printers) {
         printer.println();
      }
   }

   /**
    * Sends a square text pattern to all the printers.
    * 
    * @param height
    *           the height of the square
    */
   public void printSquare(int height) {
      for (int i = 1; i <= height; i++) {
         for (int j = 1; j <= height; j++) {
            print("x");
         }
         println();
      }
   }

   /**
    * Sends a triangle text pattern to all the printers.
    * 
    * @param height
    *           the height of the triangle
    */
   public void printTriangle(int height) {
      for (int i = 1; i <= height; i++) {
         for (int j = 1; j <= i; j++) {
            print("x");
         }
         println();
      }
   }

   /**
    * Sends a pyramid text pattern to all the printers.
    * 
    * @param height
    *           the height of the pyramid
    */
   public void printPyramid(int height) {
      for (int i = 1; i <= height; i++) {
         print(String.format("%" + (height - i + 1) + "s", "x"));
         for (int j = 2; j < (2 * i); j++) {
            print("x");
         }
         println();
      }
   }
   
   /**
    * Sends a plot of Pascal's triangle to all the printers.
    * 
    * @param height
    *           the height of the triangle
    * @param nwidth
    *           the width reserved for each number
    */
   public void printPascal(int height, int nwidth) {
      if (nwidth > 0) {
         for (int i = 1; i <= height; i++) {
            int number = 1;
            print(String.format("%" + ((height - i + 1) * nwidth) + "d", number));
            for (int j = 1; j < i; j++) {
               number = number * (i - j) / j;  // number = binomial(i - 1, j)
               print(String.format("%" + (2 * nwidth) + "d", number));
            }
            println();
         }
      }
   }
   
   /**
    * Sends a plot of Pascal's triangle to all the printers.
    * 
    * @param height
    *           the height of the triangle
    */
   public void printPascal(int height) {
      int mid = height / 2 + 1;
      int number = 1;
      for (int j = 1; j < mid; j++) {
         number = number * (height - j) / j;
      }
      int nwidth = Integer.toString(number).length();
      printPascal(height, nwidth);
   }
   
   /**
    * A small test of the TextPatternNew class.
    */
   public static void main(String[] args) {
      // Construct the text frame
      TextFrame textFrame = new TextFrame("My Text Pattern Frame");
      textFrame.setTextEditable(false);
      textFrame.start();

      // Construct the printers
      ConsolePrinter consolePrinter = new ConsolePrinter();
      TextFramePrinter framePrinter = new TextFramePrinter(textFrame);

      // Construct a text pattern generator and add its printers
      TextPattern pattern = new TextPattern();
      pattern.addPrinter(consolePrinter);
      pattern.addPrinter(framePrinter);

      // Print some text patterns
      pattern.println("a pyramid (height = 5):");
      pattern.printPyramid(5);
      pattern.println();
      pattern.println("Pascal's triangle (height = 5):");
      pattern.printPascal(5, 1);
      pattern.println();
      pattern.println("Pascal's triangle (height = 9):");
      pattern.printPascal(9);
   }

}
