import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A JFrame for sketching.
 * 
 * @author Hendrik Speleers
 * @author NMCGJ, AY 2025-2026
 */
public class SketchFrame extends GraphicsFrame {

   private static final long serialVersionUID = 1L;

   /**
    * Constructs a sketch frame with a given title.
    * 
    * @param title
    *           the frame title
    */
   public SketchFrame(String title) {
      super(title);
      addMouseComponents();
      addMenuColor();
   }

   protected JMenu menuColor;
   protected JMenuItem menuColorChooser, menuColorList[];

   protected Color[] colorCode = { 
         Color.BLACK, Color.BLUE, Color.GRAY, Color.GREEN, Color.MAGENTA, 
         Color.ORANGE, Color.PINK, Color.RED, Color.WHITE, Color.YELLOW };
   protected String[] colorName = { 
         "Black", "Blue", "Gray", "Green", "Magenta", 
         "Orange", "Pink", "Red", "White", "Yellow" };

   /**
    * Adds the color menu to the menu bar.
    */
   protected void addMenuColor() {
      ActionListener menuColorListener = new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            JMenuItem actionMenu = (JMenuItem) e.getSource();
            if (actionMenu == menuColorChooser) {
               showColorDialog();
            } else {
               setGraphicsColor(actionMenu.getForeground());
            }
         }
      };

      menuColor = new JMenu("Color");
      menuColor.setMnemonic(KeyEvent.VK_C);
      menuBar.add(menuColor);

      menuColorChooser = new JMenuItem("Choose");
      menuColorChooser.setMnemonic(KeyEvent.VK_O);
      menuColorChooser.addActionListener(menuColorListener);
      menuColor.add(menuColorChooser);

      menuColor.addSeparator();

      menuColorList = new JMenuItem[colorCode.length];
      for (int i = 0; i < colorCode.length; i++) {
         menuColorList[i] = new JMenuItem(colorName[i]);
         menuColorList[i].setForeground(colorCode[i]);
         menuColorList[i].addActionListener(menuColorListener);
         menuColor.add(menuColorList[i]);
      }
   }

   /**
    * Adds the mouse components to the frame.
    */
   protected void addMouseComponents() {
      MouseAdapter mouseAdapter = new MouseAdapter() {
         public void mousePressed(MouseEvent e) {
            startSketch(e);
         }
      };

      MouseMotionAdapter mouseMotionAdapter = new MouseMotionAdapter() {
         public void mouseDragged(MouseEvent e) {
            continueSketch(e);
         }
      };

      graphicsPanel.addMouseListener(mouseAdapter);
      graphicsPanel.addMouseMotionListener(mouseMotionAdapter);
      setSketchable(true);
   }

   protected boolean sketchable;
   protected Point sketchPoint;

   /**
    * Indicates whether sketching is allowed or not by the user.
    * 
    * @param sketchable
    *           the new status
    */
   public void setSketchable(boolean sketchable) {
      this.sketchable = sketchable;
   }

   /**
    * Starts sketching with the mouse.
    * 
    * @param e
    *           the mouse event
    */
   protected void startSketch(MouseEvent e) {
      sketchPoint = new Point(e.getX(), e.getY());
   }

   /**
    * Continues sketching with the mouse.
    * 
    * @param e
    *           the mouse event
    */
   protected void continueSketch(MouseEvent e) {
      if (sketchable) {
         Point point = new Point(e.getX(), e.getY());
         graphicsPanel.drawLine(sketchPoint, point);
         graphicsPanel.repaint();
         sketchPoint = point;
      }
   }

   /**
    * Shows a color dialog.
    */
   protected void showColorDialog() {
      Color color = JColorChooser.showDialog(this, "Choose Color",
            graphicsPanel.getDrawColor());
      setGraphicsColor(color);
   }

   /**
    * Sets the graphics color.
    * 
    * @param color
    *           the color
    */
   protected void setGraphicsColor(Color color) {
      if (color != null) {
         graphicsPanel.setDrawColor(color);
      }
   }

   /**
    * A small test of the SketchFrame class.
    */
   public static void main(String[] args) {
      SketchFrame frame = new SketchFrame("My Sketch Pad");
      frame.setMenuVisible(true);
      frame.start();

      GraphicsPanel graphics = frame.getGraphicsPanel();
      graphics.setBackground(Color.WHITE);
      graphics.setDrawColor(Color.BLACK);
      Point p1 = new Point(10, 10);
      Point p2 = new Point(20, 20);
      graphics.drawRectangle(p1, p2, false);

      Point p5 = new Point(30, 20);
      graphics.drawText("Start sketching by moving the mouse...", p5);

      Point p3 = new Point(10, 30);
      Point p4 = new Point(20, 40);
      graphics.drawRectangle(p3, p4, false);

      Point p6 = new Point(30, 40);
      int h = graphics.getHeight();
      int w = graphics.getWidth();
      graphics.drawText("Dimension of sketch pad is " + h + " x " + w, p6);
      
      graphics.setDrawColor(Color.BLUE);
      frame.setSketchable(true);
   }

}
