import java.awt.*;

/**
 * A class for drawing fractal patterns. 
 * 
 * @author Hendrik Speleers
 * @author NMCGJ, AY 2025-2026
 */
public class FractalFactory {

   protected static final double SQRT3 = Math.sqrt(3);
   protected GraphicsPanel graphics;

   /**
    * Constructs a fractal factory.
    * 
    * @param graphics
    *           the graphics panel for the fractals
    */
   public FractalFactory(GraphicsPanel graphics) {
      setGraphicsPanel(graphics);
   }

   /**
    * Gets the graphics panel.
    * 
    * @return the panel
    */
   public GraphicsPanel getGraphicsPanel() {
      return graphics;
   }

   /**
    * Sets the graphics panel.
    * 
    * @param graphics
    *           the graphics panel for the fractals
    */
   public void setGraphicsPanel(GraphicsPanel graphics) {
      this.graphics = graphics;
   }
   
   /**
    * Repaints the graphics panel.
    */
   public void paintFractal() {
      if (graphics != null) graphics.repaint();
   }

   /**
    * Clears the graphics panel.
    */
   public void clearFractal() {
      if (graphics != null) graphics.clear();
   }

   /**
    * Draws a fractal with circles.
    * 
    * @param depth
    *           the depth of the fractal
    */
   public void drawCircles(int depth) {
      if (graphics != null) {
         double xc = graphics.getWidth() / 2.0;
         double yc = graphics.getHeight() / 2.0;
         double radius = Math.min(xc / 2.0, yc) - 10.0;
         
         drawCirclesRec(xc, yc, radius, depth);
         paintFractal();
      }
   }

   /**
    * Draws a fractal with circles (recursive step).
    * 
    * @param x
    *           the x-coordinate of the center
    * @param y
    *           the y-coordinate of the center
    * @param radius
    *           the radius
    * @param depth
    *           the number of recursive steps
    */
   protected void drawCirclesRec(double x, double y, double radius, int depth) {
      Point p1 = new Point((int) (x - radius), (int) (y - radius));
      Point p2 = new Point((int) (x + radius), (int) (y + radius));
      graphics.drawOval(p1, p2);

      if (depth > 1 && radius >= 1.0) {
         drawCirclesRec(x - radius, y, radius / 2.0, depth - 1);
         drawCirclesRec(x + radius, y, radius / 2.0, depth - 1);
      }
   }

   /**
    * Draws the Sierpinski triangle.
    * 
    * @param depth
    *           the depth of the fractal
    */
   public void drawSierpinskiTriangle(int depth) {
      if (graphics != null) {
         double xc = graphics.getWidth() / 2.0;
         double yc = graphics.getHeight() / 2.0;
         double ratio = SQRT3 / 2.0;
         double len = Math.min(xc, yc / ratio) - 10.0;
   
         double x1 = xc - len;
         double y1 = yc + len * ratio;
         double x2 = xc;
         double y2 = yc - len * ratio;
         double x3 = xc + len;
         double y3 = y1;
   
         drawSierpinskiTriangleRec(x1, y1, x2, y2, x3, y3, depth);
         paintFractal();
      }
   }

   /**
    * Draws the Sierpinski triangle (recursive step).
    * 
    * @param x1
    *           the x-coordinate of the first point
    * @param y1
    *           the y-coordinate of the first point
    * @param x2
    *           the x-coordinate of the second point
    * @param y2
    *           the y-coordinate of the second point
    * @param x3
    *           the x-coordinate of the third point
    * @param y3
    *           the y-coordinate of the third point
    * @param depth
    *           the depth of the fractal
    */
   protected void drawSierpinskiTriangleRec(double x1, double y1, 
         double x2, double y2, double x3, double y3, int depth) {

      if (depth > 1 && (Math.abs(x2 - x1) + Math.abs(x3 - x2)) >= 4.0
                    && (Math.abs(y2 - y1) + Math.abs(y3 - y2)) >= 4.0) {
         double u1 = (x2 + x3) / 2.0;
         double v1 = (y2 + y3) / 2.0;
         double u2 = (x3 + x1) / 2.0;
         double v2 = (y3 + y1) / 2.0;
         double u3 = (x1 + x2) / 2.0;
         double v3 = (y1 + y2) / 2.0;

         drawSierpinskiTriangleRec(x1, y1, u3, v3, u2, v2, depth - 1);
         drawSierpinskiTriangleRec(u3, v3, x2, y2, u1, v1, depth - 1);
         drawSierpinskiTriangleRec(u2, v2, u1, v1, x3, y3, depth - 1);
         
      } else {
         Point[] pp = { new Point((int) x1, (int) y1),
                        new Point((int) x2, (int) y2), 
                        new Point((int) x3, (int) y3) };
         graphics.drawPolygon(pp, true);
      }
   }

   /**
    * Draws the Koch snowflake.
    * 
    * @param depth
    *           the depth of the fractal
    */
   public void drawKochSnowflake(int depth) {
      if (graphics != null) {
         double xc = graphics.getWidth() / 2.0;
         double yc = graphics.getHeight() / 2.0;
         double ratio = SQRT3 * 2.0 / 3.0;
         double len = Math.min(xc, yc / ratio) - 10.0;
   
         double x1 = xc - len;
         double y1 = yc + len * ratio / 2.0;
         double x2 = xc;
         double y2 = yc - len * ratio;
         double x3 = xc + len;
         double y3 = y1;
   
         drawKochCurveRec(x1, y1, x2, y2, depth);
         drawKochCurveRec(x2, y2, x3, y3, depth);
         drawKochCurveRec(x3, y3, x1, y1, depth);
         paintFractal();
      }
   }

   /**
    * Draws the Koch curve (recursive step).
    * 
    * @param x1
    *           the x-coordinate of the first point
    * @param y1
    *           the y-coordinate of the first point
    * @param x2
    *           the x-coordinate of the second point
    * @param y2
    *           the y-coordinate of the second point
    * @param depth
    *           the depth of the fractal
    */
   protected void drawKochCurveRec(double x1, double y1, double x2, double y2,
         int depth) {

      if (depth > 1 && (Math.abs(x2 - x1) + Math.abs(y2 - y1)) >= 2.0) {
         double u1 = (2.0 * x1 + x2) / 3.0;
         double v1 = (2.0 * y1 + y2) / 3.0;
         double u2 = (x1 + x2) / 2.0 + SQRT3 * (y2 - y1) / 6.0;
         double v2 = (y1 + y2) / 2.0 + SQRT3 * (x1 - x2) / 6.0;
         double u3 = (x1 + 2.0 * x2) / 3.0;
         double v3 = (y1 + 2.0 * y2) / 3.0;

         drawKochCurveRec(x1, y1, u1, v1, depth - 1);
         drawKochCurveRec(u1, v1, u2, v2, depth - 1);
         drawKochCurveRec(u2, v2, u3, v3, depth - 1);
         drawKochCurveRec(u3, v3, x2, y2, depth - 1);

      } else {
         Point p1 = new Point((int) x1, (int) y1);
         Point p2 = new Point((int) x2, (int) y2);
         graphics.drawLine(p1, p2);
      }
   }

   /**
    * A small test of the FractalFactory class.
    */
   public static void main(String[] args) {
      // Construct the graphics frame
      GraphicsFrame frame = new GraphicsFrame("My Fractal Frame");
      frame.setGraphicsDimension(500, 500);
      frame.setResizable(false);
      frame.start();

      // Read the input
      String msgFractal = "What is your favorite fractal?\n"
            + "1) fractal with circles\n" 
            + "2) Sierpinski triangle\n"
            + "3) Koch snowflake";
      int fractal = frame.showInputDialogInt(msgFractal, 1);
      if (fractal >= 1 && fractal <= 3) {
         String msgDepth = "What is the fractal depth?";
         int depth = frame.showInputDialogInt(msgDepth, 5);
         if (depth > 0) {
            // Construct the fractal factory
            GraphicsPanel graphics = frame.getGraphicsPanel();
            graphics.setDrawColor(Color.BLUE);
            FractalFactory factory = new FractalFactory(graphics);
            switch (fractal) {
            case 1:
               factory.drawCircles(depth);
               break;
            case 2:
               factory.drawSierpinskiTriangle(depth);
               break;
            case 3:
               factory.drawKochSnowflake(depth);
               break;
            }
         }
      }
   }
 
}
