import java.awt.*;

/**
 * A JFrame for showing a moving spiral.
 * 
 * @author Hendrik Speleers
 * @author NMCGJ, AY 2025-2026
 */
public class MovingSpiral extends AnimationGraphicsFrame {

   private static final long serialVersionUID = 1L;
   private int radius;
   private int diameter;
   private int scale;
   private double theta;

   /**
    * Constructs an animation frame for a moving spiral.
    */
   public MovingSpiral() {
      this(5, 10);
   }

   /**
    * Constructs an animation frame for a moving spiral.
    * 
    * @param radius
    *           the radius of the moving ball
    * @param scale
    *           the scaling factor of the spiral
    */
   public MovingSpiral(int radius, int scale) {
      super("Moving Spiral");
      setRadius(radius);
      setSpiralScale(scale);
   }

   /**
    * Sets the radius of the ball.
    * 
    * @param radius
    *           the radius of the ball
    */
   public void setRadius(int radius) {
      this.radius = (radius > 1) ? radius : 1;
      this.diameter = 2 * this.radius;
   }

   /**
    * Sets the scale of the spiral.
    * 
    * @param scale
    *           the scaling factor of the spiral
    */
   public void setSpiralScale(int scale) {
      this.scale = (scale > 1) ? scale : 1;
   }

   /**
    * Initializes a new animation (is called before the start of the animation).
    */
   protected void animateInit() {
      setAnimationDelay(200);
      theta = 0.0;
   }

   /**
    * Executes the next step in the animation.
    */
   protected void animateNext() {
      boolean inside = drawAllBalls();
      if (!inside) theta = -theta;
      theta += getSpiralStep(theta);
   }

   /**
    * Finalizes the animation (is called after the end of the animation).
    */
   protected void animateFinal() {
      theta = Double.MAX_VALUE;
      drawAllBalls();
   }

   /**
    * Draws all balls on the linear spiral.
    * 
    * @param theta
    *           the max angle value of the spiral
    * @return true if all balls inside, false otherwise
    */
   protected boolean drawAllBalls() {
      GraphicsPanel graphics = getGraphicsPanel();
      graphics.clear();
      graphics.setDrawColor(Color.BLUE);

      int border = 0; // specify here the border margin
      int width = graphics.getWidth();
      int height = graphics.getHeight();
      double abstheta = Math.abs(theta);
      double t = 0.0;
      boolean inside = true;
      while (t <= abstheta && inside) {
         int x = (int) (width / 2.0 + scale * t * Math.sin(t)) - radius;
         int y = (int) (height / 2.0 + scale * t * Math.cos(t)) - radius;
         inside = (x >= border && x <= (width - diameter - border) 
                && y >= border && y <= (height - diameter - border));
         if (inside) {
            graphics.drawOval(x, y, diameter, diameter, true);
            t += getSpiralStep(t);
         }
      }

      graphics.repaint();
      return inside;
   }

   /**
    * Computes the next angle step.
    * 
    * @param t
    *           the current angle value
    */
   protected double getSpiralStep(double t) {
      double increment = 1.25;
      if (t >= 0) {
         return Math.PI / (t + increment) * radius / scale;
      } else if (t < -(Math.PI / increment * radius / scale)) {
         double at = (Math.abs(t) + increment) / 2.0;
         double delta = at * at - Math.PI * radius / scale;
         return at - Math.sqrt(delta);
      } else {
         return Math.abs(t);
      }
   }

   /**
    * A small test of the MovingSpiral class.
    */
   public static void main(String[] args) {
      MovingSpiral frame = new MovingSpiral();
      frame.setGraphicsDimension(500, 500);
      frame.start();
   }

}
