import java.awt.*;

/**
 * A JFrame for showing a traffic light.
 * 
 * @author Hendrik Speleers
 * @author NMCGJ, AY 2025-2026
 */
public class TrafficLight extends AnimationGraphicsFrame {

   private static final long serialVersionUID = 1L;
   private int radius;
   private int diameter;
   private int buffer;
   private int status; // LIGHT: 0 = inactive, 1 = green, 2 = orange, 3 = red

   /**
    * Constructs a traffic light frame.
    */
   public TrafficLight() {
      this(50, 10);
   }

   /**
    * Constructs a traffic light frame.
    * 
    * @param radius
    *           the radius of the lights
    * @param buffer
    *           the buffer space
    */
   public TrafficLight(int radius, int buffer) {
      super("Traffic Light");
      setRadius(radius);
      setBuffer(buffer);
      setTrafficFrame();
   }

   /**
    * Sets the radius of the lights.
    * 
    * @param radius
    *           the radius of the lights
    */
   protected void setRadius(int radius) {
      this.radius = (radius > 50) ? radius : 50;
      this.diameter = 2 * this.radius;
   }

   /**
    * Sets the buffer space between the lights.
    * 
    * @param buffer
    *           the buffer space
    */
   protected void setBuffer(int buffer) {
      this.buffer = (buffer > 10) ? buffer : 10;
   }

   /**
    * Sets the dimensions of the frame according to the light radius.
    */
   protected void setTrafficFrame() {
      setGraphicsDimension(diameter + 2 * buffer, 3 * diameter + 4 * buffer);
      setResizable(false);
      animateFinal();
   }

   /**
    * Initializes a new animation (is called before the start of the animation).
    */
   protected void animateInit() {
      status = 0;
      drawLight();
      setAnimationDelay(1000);
   }

   /**
    * Executes the next step in the animation.
    */
   protected void animateNext() {
      status = (status % 3) + 1;
      drawLight();
   }

   /**
    * Finalizes the animation (is called after the end of the animation).
    */
   protected void animateFinal() {
      status = 0;
      drawLight();
   }

   /**
    * Draws the traffic light.
    */
   protected void drawLight() {
      GraphicsPanel graphics = getGraphicsPanel();
      graphics.clear();
      graphics.setBackground(Color.BLACK);
      graphics.setDrawColor((status == 3) ? Color.RED : Color.GRAY);
      graphics.drawOval(buffer, buffer, diameter, diameter, true);
      graphics.setDrawColor((status == 2) ? Color.ORANGE : Color.GRAY);
      graphics.drawOval(buffer, diameter + 2 * buffer, diameter, diameter, true);
      graphics.setDrawColor((status == 1) ? Color.GREEN : Color.GRAY);
      graphics.drawOval(buffer, 2 * diameter + 3 * buffer, diameter, diameter, true);
      graphics.repaint();
   }

   /**
    * A small test of the TrafficLight class.
    */
   public static void main(String[] args) {
      TrafficLight frame = new TrafficLight();
      frame.start();
   }

}
